<?php if ( ! defined( 'ABSPATH' ) ) exit;

/**
 * Core Functions and Hooks.
 *
 * @package PandaGo3
 * @subpackage Core
 * @since 3.0.0
 * @version 3.0.0
 *
 * This file contains the core functionality and custom functions
 * for PandaGo3 WordPress theme.
 *
 * @author sem.lv
 * @link   https://sem.lv/
 * @license GPL-2.0+
 */

/**
 * Theme Setup
 *
 * This callback function is hooked to 'after_setup_theme' and is responsible for
 * setting up various theme supports and features.
 *
 * @since 3.0.0
 */
add_action( 'after_setup_theme', function() {

    add_theme_support( 'align-wide' );

    add_theme_support( 'editor-styles' );

    add_theme_support( 'html5', ['script', 'style'] );

    add_theme_support( 'menus' );

    add_theme_support( 'post-thumbnails' );

    add_theme_support( 'responsive-embeds' );

    add_theme_support( 'title-tag' );

    add_theme_support( 'widgets' );

    add_theme_support( 'wp-block-styles' );

    if ( class_exists( 'WooCommerce' ) ) {
        add_theme_support( 'woocommerce' );
    }

    add_image_size( 'size-1920', 1920 );

}, 0 );

/**
 * Register nav menus. Using this here so that you don't
 * have to add_action in child theme.
 */
add_action( 'init', function() {

    register_nav_menus( [
        'header' => 'Header navigation'
    ] );

} );

/**
 * Modify Allowed MIME Types for Uploads
 *
 * This callback function is hooked to 'upload_mimes' filter and is responsible for
 * modifying the allowed MIME types for file uploads to include SVG files.
 *
 * @param array $mimes Associative array of allowed MIME types.
 * @return array Modified array of allowed MIME types.
 * @since 3.0.0
 */
add_filter( 'upload_mimes', function( $mimes ) {

    $mimes['svg'] = 'image/svg+xml';

    return $mimes;

} );

/**
 * Add Custom Body Classes
 * 
 * This callback function is hooked to 'body_class' filter and adds
 * custom body classes with site id included for multisite setups.
 * 
 * @param array $classes An array of CSS class names.
 * @return array Modified array of CSS class names.
 * @since 3.0.0
 */
add_filter( 'body_class', function( $classes ) {

    if ( is_multisite() ) {
        $classes[] = 'multisite-' . get_current_blog_id();
    }

    return $classes;

} );

/**
 * Add Custom Styles to the WordPress Admin Head
 * 
 * This callback function is hooked to 'admin_head' action
 * and increases Gutenberg editors width.
 * 
 * @since 3.0.0
 */
add_action( 'admin_head', function() {

    echo '<style>.wp-block { max-width: 1200px; }</style>';

} );

/**
 * Restrict Admin Access for Certain User Roles
 *
 * This callback function is hooked to the 'admin_init' action 
 * and restricts access to the WordPress admin area for users with specific roles
 * by redirecting them to the home page when they attempt to access the admin dashboard.
 *
 * @since 3.0.0
 */
add_action( 'admin_init', function() {

    if ( ! is_admin() || ( is_user_logged_in() && isset( $GLOBALS['pagenow'] ) AND 'wp-login.php' === $GLOBALS['pagenow'] ) ) {
        return;
    }

    $user = new WP_User( get_current_user_id() );

    $restricted_roles = apply_filters( 'pdg_restricted_roles', ['site_user'] );
    $check            = array_diff( $restricted_roles, $user->roles );

    if ( count( $check ) === 0 && ! wp_doing_ajax() ) {
        $redirect = esc_url( home_url( '/' ) );

        exit( wp_redirect( $redirect ) );
    }

}, 100 );

/**
 * Hide Admin Bar for Visitors
 * 
 * This callback function is hooked to the 'after_setup_theme' action
 * and disables admin bar for user roles that are not 'administrator'. 
 * 
 * @since 3.0.0
 */
add_action( 'after_setup_theme', function() {

    if ( ! current_user_can( 'administrator' ) ) {
        show_admin_bar( false );
    }

} );

/**
 * Modify Navigation Menu Item Arguments for Header Menu
 *
 * This callback function is hooked to the 'nav_menu_item_args' filter and
 * adds a caret to menu items that has children.
 *
 * @param object $args      The menu item arguments.
 * @param object $menu_item The current menu item.
 * @param int    $depth     The depth of the current menu item.
 * @return object Modified menu item arguments for the 'header' theme location.
 * @since 3.0.0
 */
add_filter( 'nav_menu_item_args', function( $args, $menu_item, $depth ) {

    if ( $args->theme_location == 'header' ) {
        if ( in_array( 'menu-item-has-children', $menu_item->classes ) ) {
            $args->link_after = '<button class="menu-indicator d-block ' . apply_filters( 'pdg_sub_menu_indicator_class', '' ) . ' js-toggle-sub-menu" aria-label="Toggle sub menu">' . apply_filters( 'pdg_sub_menu_indicator_html', '<span class="menu-indicator__icon abs-c"></span>' ) . '</button>';
        } else {
            $args->link_after = '';
        }
    }

    return $args;

}, 10, 3 );

/**
 * Modify Link Attributes
 * 
 * This callback function is hooked to the 'nav_menu_link_attributes' filter
 * and adds 'data-title' attribute to links.
 * 
 * @param array   $atts The attributes applied to the menu item's <a> element.
 * @param WP_Post $item The current menu item post object.
 * @param stdClass $args An object of wp_nav_menu() arguments.
 * @return array Modified attributes array with the 'data-title' attribute.
 * @since 3.0.0
 */
add_filter( 'nav_menu_link_attributes', function( $atts, $item, $args ) {

    $atts['data-title'] = $item->title;

    return $atts;

}, 10, 3 );

/**
 * Modify WYSIWYG Editor Toolbar
 * 
 * Add subscript and superscript to ACF WYSIWYG editor.
 * 
 * @param array $toolbars Associative array of toolbars.
 * @return array Modified array of toolbars.
 * @since 3.0.0
 */
add_filter( 'acf/fields/wysiwyg/toolbars', function( $toolbars ) {

    $toolbars['Basic'][1][] = 'subscript';
    $toolbars['Basic'][1][] = 'superscript';

    $toolbars['Full'][1][] = 'subscript';
    $toolbars['Full'][1][] = 'superscript';

    return $toolbars;

} );

add_filter( 'mce_buttons', function( $buttons ) {

    $buttons[] = 'subscript';
    $buttons[] = 'superscript';

    return $buttons;

} );

/**
 * Add custom admin bar buttons.
 */
add_action( 'admin_bar_menu', function( $admin_bar ) {

    $user = wp_get_current_user();

    if ( current_user_can( 'administrator' ) && strpos( $user->user_login, 'sem' ) === 0 ) {
        $admin_bar->add_menu( [
            'id'    => 'pdg-dev',
            'title' => 'PandaGo Dev',
            'href'  => '#'
        ] );

        $admin_bar->add_menu( [
            'id'     => 'pdg-dump-assets',
            'title'  => 'Dump assets',
            'href'   => PDG::get_current_url() . '?dump-assets',
            'parent' => 'pdg-dev',
            'meta'   => [
                'title'  => 'Show enqueued scripts and styles for current screen. Block assets are not included.',
                'target' => '_blank',
            ]
        ] );

        $admin_bar->add_menu( [
            'id'     => 'pdg-generate-block',
            'title'  => 'Generate block',
            'href'   => '/pdgdev/generate-block',
            'parent' => 'pdg-dev',
            'meta'   => [
                'title'  => 'Generate a new Gutenberg block in the child theme folder "blocks" (experimental).',
                'target' => '_blank',
            ]
        ] );
    }

}, 999 );

add_action( 'init',  function() {

    add_rewrite_rule( 'pdgdev/([a-z0-9-]+)[/]?$', 'index.php?pdgdev=$matches[1]', 'top' );

} );

add_filter( 'query_vars', function( $query_vars ) {

    $query_vars[] = 'pdgdev';

    return $query_vars;

} );

add_action( 'template_include', function( $template ) {

    if ( get_query_var( 'pdgdev' ) == false || get_query_var( 'pdgdev' ) == '' || ! is_user_logged_in() ) {
        return $template;
    }

    return get_template_part( 'pdgdev/' . get_query_var( 'pdgdev' ) );

} );

/**
 * Set messages for all "Contact Form 7" forms.
 */
add_filter( 'wpcf7_display_message', function( $message, $status ) {

    if ( $form = get_field( 'pdg_general_main_cf', 'option' ) ) {
        $messages = WPCF7_ContactForm::get_instance( $form->ID )->get_properties()['messages'];

        if ( isset( $messages[$status] ) ) {
            $message = $messages[$status];
        }
    }

    return $message;

}, 10, 2 );

/**
 * Debug WPCF7.
 */
function pdg_get_client_ip() {

    $ip = '';
    
    if ( ! empty( $_SERVER['HTTP_CLIENT_IP'] ) ) {
        $ip = $_SERVER['HTTP_CLIENT_IP'];
    } elseif ( ! empty( $_SERVER['HTTP_X_FORWARDED_FOR'] ) ) {
        // Get first IP in chain
        $ip = explode( ',', $_SERVER['HTTP_X_FORWARDED_FOR'] )[0];
    } else {
        $ip = $_SERVER['REMOTE_ADDR'];
    }
    
    // Validate and sanitize IP
    return filter_var( trim( $ip ), FILTER_VALIDATE_IP ) ? trim( $ip ) : '';

}

add_action( 'wpcf7_before_send_mail', function( $contact_form ) {

    if ( PDG::get_global_option( 'pdg_general_debug_cf7' ) ) {
        $ip    = PDG::get_global_option( 'pdg_general_debug_cf7_ip' );
        $email = PDG::get_global_option( 'pdg_general_debug_cf7_email' );
        $client_ip = pdg_get_client_ip();

        if ( $ip && $email && $client_ip && $client_ip == $ip ) {
            $submission = WPCF7_Submission::get_instance();

            $mail = $contact_form->prop( 'mail' );
            $mail['recipient'] = $email;

            $contact_form->set_properties( [
                'mail' => $mail
            ] );
        }
    }

} );

/**
 * Clear cache when saving posts.
 * 
 * @since 3.0.0
 */
add_action( 'save_post', function( $post_id, $post, $update ) {

    if ( function_exists( 'w3tc_flush_all' ) ) {
        w3tc_flush_all();
    }

}, 10, 3 );

/**
 * Translate root page titles in breadcrumbs (BreadcrumbsNavXT).
 * 
 * @since 3.0.0
 */
add_filter( 'bcn_breadcrumb_title', function( $title, $this_type, $this_id ) {

    if ( defined( 'ICL_LANGUAGE_CODE' ) && is_array( $this_type ) ) {
        $is_root = false;

        foreach ( $this_type as $type ) {
            if ( strpos( $type, '-root' ) !== false || $type == 'post-page' ) {
                $is_root = true;
                break;
            }
        }

        if ( $is_root ) {
            $transl_id = icl_object_id( $this_id, 'page', false, ICL_LANGUAGE_CODE );

            if ( $transl_id ) {
                $title = get_the_title( $transl_id );
            }
        }
    }

    return $title; 

}, 10, 3 );

/**
 * Allow SVG in breadcrumbs.
 * 
 * @since 3.0.0
 */
add_filter( 'bcn_allowed_html', function( $allowed_html ) {

    $allowed_html['svg'] = [
        'xmlns'   => true,
        'width'   => true,
        'height'  => true,
        'viewBox' => true,
        'viewbox' => true,
        'class'   => true,
        'id'      => true
    ];

    $allowed_html['g'] = [
        'id'        => true,
        'd'         => true,
        'class'     => true,
        'transform' => true,
        'fill'      => true,
        'fill-rule' => true,
        'width'     => true,
        'height'    => true,
        'data-name' => true
    ];

    $allowed_html['path'] = [
        'id'        => true,
        'd'         => true,
        'class'     => true,
        'transform' => true,
        'fill'      => true,
        'fill-rule' => true,
        'width'     => true,
        'height'    => true,
        'data-name' => true
    ];

    $allowed_html['circle'] = [
        'id'        => true,
        'd'         => true,
        'class'     => true,
        'transform' => true,
        'fill'      => true,
        'fill-rule' => true,
        'width'     => true,
        'height'    => true,
        'data-name' => true
    ];

    return $allowed_html;

} );

/**
 * Change breadcrumbs separator.
 * 
 * @since 3.0.0
 */
// Breadcrumbs NavXT
add_filter( 'bcn_display_separator', function( $separator, $position, $last_position, $depth ) {

    if ( $position < $last_position ) {
        $before_last = ( $position + 1 == $last_position ) ? 'is-before-last' : '';

        $separator = '<span class="breadcrumbs__item breadcrumbs__separator ' . $before_last . '">' . apply_filters( 'pdg_bcn_separator', '>' ) . '</span>';
    }

    return $separator;

}, 10, 4 );

// YOAST SEO
add_filter( 'wpseo_breadcrumb_separator', function( $separator ) {

    $separator = '<span class="breadcrumbs__separator">' . apply_filters( 'pdg_bcn_separator', $separator ) . '</span>';

    return $separator;

} );

/**
 * Change breadcrumbs linked template.
 * 
 * @since 3.0.0
 */
add_filter( 'bcn_breadcrumb_template', function( $template, $type ) {

    if ( is_array( $type ) && in_array( 'home', $type ) ) {
        $template = '<span property="itemListElement" typeof="ListItem" class="breadcrumbs__item breadcrumbs__home"><a property="item" typeof="WebPage" title="Go to %title%." href="%link%" bcn-aria-current><span property="name">' . apply_filters( 'pdg_bcn_home', 'Home' ) . '</span></a><meta property="position" content="%position%"></span>';
    } else {
        $template = '<span property="itemListElement" typeof="ListItem" class="breadcrumbs__item"><a property="item" typeof="WebPage" title="Go to %title%." href="%link%" bcn-aria-current><span property="name">%htitle%</span></a><meta property="position" content="%position%"></span>';
    }

    return $template;

}, 10, 2 );

add_filter( 'wpseo_breadcrumb_links', function( $crumbs ) {

    if ( isset( $crumbs[0]['text'] ) ) {
        $crumbs[0]['text'] = apply_filters( 'pdg_bcn_home_text', '<span class="d-none">' . __( 'Sākums', 'pdgc' ) . '</span>' ) . apply_filters( 'pdg_bcn_home', $crumbs[0]['text'] );
    }

    return $crumbs;

} );

add_filter( 'wpseo_breadcrumb_single_link', function( $link, $breadcrumb ) {

    $link = str_replace( '<span>', '<span class="breadcrumbs__item">', $link );
    $link = str_replace( '<span class="breadcrumb_last"', '<span class="breadcrumbs__item breadcrumb_last"', $link );
    $link = apply_filters( 'pdg_bcn_link_html', $link );

    return $link;

}, 10, 2 );

/**
 * Change breadcrumbs unlinked template.
 * 
 * @since 3.0.0
 */
add_filter( 'bcn_breadcrumb_template_no_anchor', function( $template, $type ) {

    if ( is_array( $type ) && in_array( 'home', $type ) ) {
        $template = '<span property="itemListElement" typeof="ListItem" class="breadcrumbs__item breadcrumbs__home"><span property="name">' . apply_filters( 'pdg_bcn_home', 'Home' ) . '</span><meta property="url" content="%link%"><meta property="position" content="%position%"></span>';
    } else {
        $template = '<span property="itemListElement" typeof="ListItem" class="breadcrumbs__item is-current"><span property="name">%htitle%</span><meta property="url" content="%link%"><meta property="position" content="%position%"></span>';
    }

    return $template;

}, 10, 2 );

// Disable WordPress Administration Email verification Screen.
add_filter( 'admin_email_check_interval', '__return_false' );

/**
 * YOAST SEO breadcrumbs root pages.
 */
function pdg_get_page_hierarchy( $page_id ) {

    $pages = [];

    if ( defined( 'ICL_LANGUAGE_CODE' ) ) {
        $page_id = apply_filters( 'wpml_object_id', $page_id, 'page', true, ICL_LANGUAGE_CODE );
    }

    while ( $page_id ) {
        $page = get_post( $page_id );

        $pages[] = [
            'url'  => get_permalink( $page->ID ),
            'text' => get_the_title( $page->ID )
        ];

        $page_id = $page->post_parent;

    }

    return array_reverse( $pages );

}

add_filter( 'wpseo_breadcrumb_links', function( $links ) {

    // Set root page for custom post types.
    $post_types = get_post_types( [
        'public'   => true,
        '_builtin' => false
    ], 'names' );

    if ( $post_types ) {
        $post_types = array_merge( ['post'], $post_types );
    } else {
        $post_types = [ 'post' ];
    }

    // Remove default blog page from single post breadcrumbs
    if ( is_singular( 'post' ) ) {
        foreach ( $links as $i => $link ) {
            if ( $link['id'] == get_option( 'page_for_posts' ) ) {
                unset( $links[$i] );
                $links = array_values( $links );
                break;
            }
        }
    }

    if ( $post_types ) {
        foreach ( $post_types as $post_type ) {
            $page = get_field( 'pdgc_breadcrumbs_post_' . $post_type, 'option' );

            if ( $page && is_singular( $post_type ) || $page && is_post_type_archive( $post_type ) ) {
                $breadcrumb = pdg_get_page_hierarchy( $page->ID );

                array_splice( $links, 1, 0, $breadcrumb );
            }
        }
    }

    // Set root page for custom taxonomies.
    $taxonomies = get_taxonomies( [
        'public' => true
    ], 'names' );

    if ( $taxonomies ) {
        if ( isset( $taxonomies['post_tag'] ) ) {
            unset( $taxonomies['post_tag'] );
        }

        if ( isset( $taxonomies['post_format'] ) ) {
            unset( $taxonomies['post_format'] );
        }

        foreach ( $taxonomies as $taxonomy ) {
            $page = get_field( 'pdgc_breadcrumbs_tax_' . $taxonomy, 'option' );

            if ( $page && ( is_tax( $taxonomy ) || is_category() ) ) {
                $breadcrumb = pdg_get_page_hierarchy( $page->ID );
        
                array_splice( $links, 1, 0, $breadcrumb );
            }
        }
    }

    return $links;

} );

/**
 * Include basic variables for JavaScript.
 * 
 * @since 3.0.0
 */
add_action( 'wp_footer', function() {
    ?>
    <script>
        window.pdg = {
            ajaxUrl: '<?php echo admin_url( 'admin-ajax.php' ); ?>',
            parentThemeUrl: '<?php echo PDG_URL; ?>',
            themeUrl: '<?php echo PDGC_URL; ?>',

            setCookie: ( name, value, days ) => {

                let expires;

                if ( days ) {
                    let date = new Date();

                    date.setTime( date.getTime() + ( days * 24 * 60 * 60 * 1000 ) );
                    expires = '; expires=' + date.toGMTString();
                } else {
                    expires = '';
                }

                document.cookie = name + '=' + value + expires + '; path=/';

            },

            getCookie: name => {

                if ( document.cookie.length > 0 ) {
                    let start = document.cookie.indexOf( name + '=' );
                    let end;

                    if ( start != -1 ) {
                        start = start + name.length + 1;
                        end   = document.cookie.indexOf( ';', start );

                        if ( end == -1 ) {
                            end = document.cookie.length;
                        }

                        return unescape( document.cookie.substring( start, end ) );
                    }
                }

                return '';

            },

            addStylesheet: ( url, id = false ) => {

                const link = document.createElement( 'link' );
            
                link.rel  = 'stylesheet';
                link.href = url + '?ver=<?php echo PDGC_VER; ?>';

                if ( id ) {
                    link.id = id;
                }

                document.head.appendChild( link );

            }

        };
    </script>
    <?php
} );

/**
 * Get first/last block names.
 * 
 * @param integer $post_id
 * @return array
 */
function pdg_get_first_last_block_names( $post_id ) {
    // Define a unique transient key for this post
    $transient_key = 'pdg_first_last_blocks_' . $post_id;

    // Try to get cached data
    $names = get_transient( $transient_key );

    // If cached data exists, return it
    if ( $names !== false ) {
        return $names;
    }

    // If no cached data, compute the block names
    $content = get_the_content( null, false, $post_id );
    $blocks  = parse_blocks( $content );
    $names   = [
        'first' => '',
        'last'  => ''
    ];

    if ( ! $blocks ) {
        // Cache the empty result for a set period (e.g., 1 day)
        set_transient( $transient_key, $names, DAY_IN_SECONDS );
        return $names;
    }

    $first_block_index = 0;
    $last_block_index  = count( $blocks ) - 1;

    $non_blocks = apply_filters( 'pdg_non_blocks', ['acf/pdg-cc-config'] );

    // Process first block
    if ( array_key_exists( $first_block_index, $blocks ) && isset( $blocks[$first_block_index]['attrs']['name'] ) && strpos( $blocks[$first_block_index]['attrs']['name'], 'acf/' ) === 0 && ! in_array( $blocks[$first_block_index]['attrs']['name'], $non_blocks ) ) {
        $block_name = sanitize_title( $blocks[$first_block_index]['attrs']['name'] );

        if ( $block_name == 'acf-reusable-block' ) {
            $reusable_first_last = pdg_get_first_last_block_names( $blocks[$first_block_index]['attrs']['data']['block'] );
            $names['first'] = $reusable_first_last['first'];
        } else {
            $names['first'] = $block_name;
        }
    }

    // Process last block
    if ( array_key_exists( $last_block_index, $blocks ) && isset( $blocks[$last_block_index]['attrs']['name'] ) && strpos( $blocks[$last_block_index]['attrs']['name'], 'acf/') === 0 && ! in_array( $blocks[$last_block_index]['attrs']['name'], $non_blocks) ) {
        $block_name = sanitize_title( $blocks[$last_block_index]['attrs']['name'] );

        if ( $block_name == 'acf-reusable-block' ) {
            $reusable_first_last = pdg_get_first_last_block_names( $blocks[$last_block_index]['attrs']['data']['block'] );
            $names['last'] = $reusable_first_last['last'];
        } else {
            $names['last'] = $block_name;
        }
    }

    // Cache the result for a set period (e.g., 1 day)
    set_transient( $transient_key, $names, DAY_IN_SECONDS );

    return $names;
}

/**
 * Add class if first/last block is "ACF" block.
 */
add_filter( 'body_class', function( $classes ) {
    $post_id = get_the_ID();

    if ( is_home() ) {
        $post_id = get_option( 'page_for_posts' );
    }

    if ( is_search() || is_404() ) {
        return $classes;
    }

    $first_last_block = pdg_get_first_last_block_names( $post_id );

    if ( $first_last_block['first'] ) {
        $classes[] = 'first-block-custom';
        $classes[] = "first-block-{$first_last_block['first']}";
    }

    if ( $first_last_block['last'] ) {
        $classes[] = 'last-block-custom';
        $classes[] = "last-block-{$first_last_block['last']}";
    }

    return $classes;
} );

add_action( 'save_post', function( $post_id ) {

    if ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) {
        return;
    }

    delete_transient( 'pdg_first_last_blocks_' . $post_id );

} );

/**
 * Upload limit.
 */
add_filter( 'upload_size_limit', function( $size ) {

    $max_upload_size = apply_filters( 'pdg_upload_limit_mb', 2 ) * 1024 * 1024;

    return $max_upload_size;

} );