<?php if ( ! defined( 'ABSPATH' ) ) exit;

/**
 * Cleanup environment.
 *
 * @package PandaGo3
 * @subpackage Core
 * @since 3.0.0
 * @version 3.0.0
 *
 * This file contains functions and hooks that removes unnecessary
 * WordPress features.
 *
 * @author sem.lv
 * @link   https://sem.lv/
 * @license GPL-2.0+
 */

/**
 * Remove WordPress Assets.
 * 
 * This callback function is hooked to 'wp_enqueue_scripts' removes
 * unnecessary styles and scripts added by WordPress.
 * 
 * @since 3.0.0
 */
add_action( 'wp_enqueue_scripts', function() {

    if ( ! is_admin() ) {
        $handles = ['wp-block-library', 'wp-block-library-theme', 'classic-theme-styles', 'global-styles', 'wpml-blocks', 'wpml-legacy-horizontal-list-0'];

        foreach ( $handles as $handle ) {
            wp_deregister_style( $handle );
            wp_dequeue_style( $handle );
        }
    }

} );

/**
 * Remove jQuery migrate.
 */
add_action( 'wp_default_scripts', function( $scripts ) {

    if ( ! is_admin() && ! empty( $scripts->registered['jquery'] ) ) {
        $scripts->registered['jquery']->deps = array_diff(
            $scripts->registered['jquery']->deps,
            [ 'jquery-migrate' ]
        );
    }

} );

/**
 * Move jQuery to body.
 */
function pdg_move_jquery_to_footer() {

    wp_scripts()->add_data( 'jquery', 'group', 1 );
    wp_scripts()->add_data( 'jquery-core', 'group', 1 );
    wp_scripts()->add_data( 'jquery-migrate', 'group', 1 );

}
add_action( 'wp_enqueue_scripts', 'pdg_move_jquery_to_footer' );

/**
 * Remove comments by default.
 */
function pdg_should_disable_comments() {

    return apply_filters( 'pdg_disable_comments', true );

}

function pdg_init_comment_disabling() {

    if ( ! pdg_should_disable_comments() ) {
        return;
    }

    add_action( 'admin_menu', 'pdg_remove_comments_menu' );
    add_action( 'init', 'pdg_disable_comments_status' );
    add_action( 'init', 'pdg_block_wp_comments_post' );
    add_filter( 'comments_array', 'pdg_hide_existing_comments', 10, 2 );
    add_filter( 'xmlrpc_enabled', '__return_false' );
    add_filter( 'rest_endpoints', 'pdg_remove_comments_rest_endpoints' );
    add_filter( 'wp_headers', 'pdg_remove_comments_rest_headers' );

}
add_action( 'after_setup_theme', 'pdg_init_comment_disabling' );

function pdg_remove_comments_menu() {

    remove_menu_page( 'edit-comments.php' );

    remove_meta_box( 'dashboard_recent_comments', 'dashboard', 'normal' );
    
    if ( is_admin_bar_showing() ) {
        remove_action( 'admin_bar_menu', 'wp_admin_bar_comments_menu', 60 );
    }

}

function pdg_disable_comments_status() {

    $post_types = get_post_types();

    foreach ( $post_types as $post_type ) {
        if ( post_type_supports( $post_type, 'comments' ) ) {
            remove_post_type_support( $post_type, 'comments' );
            remove_post_type_support( $post_type, 'trackbacks' );
        }
    }

}

function pdg_hide_existing_comments( $comments ) {

    return [];

}

function pdg_block_wp_comments_post() {

    if ( isset( $_SERVER['REQUEST_URI'] ) && strpos( $_SERVER['REQUEST_URI'], 'wp-comments-post.php' ) !== false ) {
        wp_die( 'Comments are disabled.', 'Comment Blocked', [ 'response' => 403 ] );
    }

    if ( basename( $_SERVER['PHP_SELF'] ) === 'wp-comments-post.php' ) {
        wp_die( 'Comments are disabled.', 'Comment Blocked', [ 'response' => 403 ] );
    }

}

function pdg_remove_comments_rest_endpoints( $endpoints ) {

    return array_filter( $endpoints, function( $key ) {
        return ! preg_match( '#^/wp/v2/comments#', $key );
    }, ARRAY_FILTER_USE_KEY );
    
}

function pdg_remove_comments_rest_headers( $headers ) {

    if ( strpos( $_SERVER['REQUEST_URI'], 'wp-comments-post.php' ) !== false ) {
        $headers['X-Robots-Tag'] = 'noindex';
        $headers['X-Content-Type-Options'] = 'nosniff';
    }

    return $headers;

}

// oEmbed links in head.
remove_action( 'wp_head', 'rest_output_link_wp_head' );
remove_action( 'wp_head', 'wp_oembed_add_discovery_links' );
remove_action( 'template_redirect', 'rest_output_link_header', 11 );

// Display meta tag with WP version.
remove_action( 'wp_head', 'wp_generator' );

// Display DNS prefetch link.
remove_action( 'wp_head', 'wp_resource_hints', 2 );

// Display the links to the extra feeds such as category feeds
remove_action( 'wp_head', 'feed_links_extra', 3 );

// Display the links to the general feeds: Post and Comment Feed
remove_action( 'wp_head', 'feed_links', 2 );

// Display the link to the Really Simple Discovery service endpoint, EditURI link
remove_action( 'wp_head', 'rsd_link' );

// Display the link to the Windows Live Writer manifest file.
remove_action( 'wp_head', 'wlwmanifest_link' );

// Remove index link
remove_action( 'wp_head', 'index_rel_link' ); 

// Remove prev link
remove_action( 'wp_head', 'parent_post_rel_link', 10, 0 ); 

// Remove start link
remove_action( 'wp_head', 'start_post_rel_link', 10, 0 );

// Display relational links for the posts adjacent to the current post.
remove_action( 'wp_head', 'adjacent_posts_rel_link', 10, 0 );

remove_action( 'wp_head', 'adjacent_posts_rel_link_wp_head', 10, 0 );
remove_action( 'wp_head', 'rel_canonical' );
remove_action( 'wp_head', 'wp_shortlink_wp_head', 10, 0 );

// Remove emoji scripts and styles
remove_action( 'wp_head', 'print_emoji_detection_script', 7 ); 
remove_action( 'admin_print_scripts', 'print_emoji_detection_script' );
remove_action( 'wp_print_styles', 'print_emoji_styles' );
remove_action( 'admin_print_styles', 'print_emoji_styles' );

// Allow shortcodes in Dynamic Sidebar
add_filter( 'widget_text', 'do_shortcode' ); 

// Remove <p> tags in Dynamic Sidebars
add_filter( 'widget_text', 'shortcode_unautop' );

// Remove <p> and <br> tags from Contact Form 7
add_filter( 'wpcf7_autop_or_not', '__return_false' );

// Remove auto <p> tags in Excerpt (Manual Excerpts only)
add_filter( 'the_excerpt', 'shortcode_unautop' );

// Allows Shortcodes to be executed in Excerpt (Manual Excerpts only)
add_filter( 'the_excerpt', 'do_shortcode' );

// Remove <p> tags from Excerpt altogether
remove_filter( 'the_excerpt', 'wpautop' );

// Disallows WP to convert quotes inside the_content() function. Removeing this because it breaks output from json_encode() inside the_content().
remove_filter( 'the_content', 'wptexturize' );

// Remove <svgs> which are loaded after opening body tag.
remove_action( 'wp_body_open', 'wp_global_styles_render_svg_filters' );

remove_action( 'xmlrpc_rsd_apis', 'rest_output_rsd' );
remove_action( 'template_redirect', 'rest_output_link_header', 11, 0 );