<?php if ( ! defined( 'ABSPATH' ) ) exit;

class PDG
{

    public static function img( $image, $size = 'full', $args = [] ) {

        // Set function attributes.
        $defaults = [
            'class'      => [],
            'title'      => false,
            'lazy'       => false,
            'dimensions' => true,
            'srcset'     => [],
            'sizes'      => ''
        ];

        $args = array_merge( $defaults, $args );

        // Determin image ID.
        if ( is_array( $image ) && isset( $image['ID'] ) ) {
            $id = $image['ID'];
        } else {
            $id = $image;
        }

        // Get full image source for checks.
        $full_src = wp_get_attachment_image_src( $id, 'full' );

        if ( ! $full_src ) {
            return;
        }

        $full_src = $full_src[0];

        // Get file extension.
        if ( function_exists( 'pathinfo' ) ) {
            $ext = strtolower( pathinfo( $full_src, PATHINFO_EXTENSION ) );
        } else {
            $last_dot_position = strrpos( $full_src, '.' );

            if ( $last_dot_position !== false ) {
                $ext = strtolower( substr( $full_src, $last_dot_position + 1 ) );
            } else {
                return;
            }
        }

        // Get image source.
        $wp_src = wp_get_attachment_image_src( $id, $size );

        $src    = $wp_src[0];
        $width  = $wp_src[1];
        $height = $wp_src[2];

        // Get image srcset.
        $srcset = '';

        if ( $args['srcset'] ) {
            $srcset = 'srcset="';
            $count  = count( $args['srcset'] );

            $i = 0;

            foreach ( $args['srcset'] as $screen_size => $image_source ) {
                $srcset .= $image_source . ' ' . $screen_size;

                if ( $i != $count - 1 ) {
                    $srcset .= ',';
                }

                $i++;
            }

            $srcset .= '"';

            if ( $args['sizes'] ) {
                $srcset .= ' sizes="' . $args['sizes'] . '"';
            }
        }

        // Prepare 'width' and 'height' attributes.
        if ( $ext == 'svg' ) {
            $width  = '';
            $height = '';
        } else {
            $width  = "width=\"{$width}\"";
            $height = "height=\"{$height}\"";
        }

        if ( ! $args['dimensions'] ) {
            $width  = '';
            $height = '';
        }

        // Get image 'title' and 'alt' attributes.
        $title = '';

        if ( is_array( $image ) ) {
            if ( $args['title'] ) {
                $title = $image['title'];
            }

            $alt = $image['alt'];

            $alt = $alt ?: $image['caption'] ?: $image['description'] ?: $image['title'];
        } else {
            $attachment = get_post( $id );

            if ( $args['title'] ) {
                $title = $attachment->post_excerpt ? $attachment->post_excerpt : $attachment->post_title;
            }

            $alt = get_post_meta( $attachment->ID, '_wp_attachment_image_alt', true );

            $alt = $alt ?: $attachment->post_excerpt ?: $attachment->post_title;
        }

        // Prepare "lazy" attribute.
        $lazy = '';

        if ( $args['lazy'] ) {
            $lazy = 'loading="lazy"';
        }

        // Prepare class name.
        $class = '';

        if ( $args['class'] ) {
            if ( is_array( $args['class'] ) ) {
                $class = 'class="' . implode( ' ', $args['class'] ) . '"';
            } else {
                $class = 'class="' . $args['class'] . '"';
            }
        }

        // Echo complete <img> tag.
        echo "<img src=\"{$src}\" {$srcset} {$class} title=\"{$title}\" alt=\"{$alt}\" {$width} {$height} {$lazy} decoding=\"async\" role=\"img\" aria-label=\"{$alt}\" aria-describedby=\"image-description\">";
    }

    public static function svg_img( $image, $size = 'full', $args = [] ) {

        $allowed_svg_tags = [
            'svg' => [
                'xmlns' => true,
                'width' => true,
                'height' => true,
                'viewbox' => true,
                'viewBox' => true,
                'preserveaspectratio' => true,
                'class' => true,
                'id' => true,
                'fill' => true,
                'stroke' => true,
            ],
            'g' => [
                'fill' => true,
                'stroke' => true,
                'transform' => true,
                'class' => true,
            ],
            'path' => [
                'd' => true,
                'fill' => true,
                'stroke' => true,
                'stroke-width' => true,
                'class' => true,
            ],
            'circle' => [
                'cx' => true,
                'cy' => true,
                'r' => true,
                'fill' => true,
                'stroke' => true,
            ],
            'rect' => [
                'x' => true,
                'y' => true,
                'width' => true,
                'height' => true,
                'fill' => true,
                'stroke' => true,
            ],
            'polygon' => [
                'points' => true,
                'fill' => true,
                'stroke' => true,
            ],
            'polyline' => [
                'points' => true,
                'fill' => true,
                'stroke' => true,
            ],
            'line' => [
                'x1' => true,
                'y1' => true,
                'x2' => true,
                'y2' => true,
                'stroke' => true,
            ],
            'ellipse' => [
                'cx' => true,
                'cy' => true,
                'rx' => true,
                'ry' => true,
                'fill' => true,
                'stroke' => true,
            ],
            'text' => [
                'x' => true,
                'y' => true,
                'fill' => true,
                'font-size' => true,
            ],
            'defs' => [],
            'style' => [
                'type' => true,
            ],
        ];

        // Set function attributes.
        $defaults = [
            'class'  => [],
            'title'  => false,
            'lazy'   => false,
            'srcset' => []
        ];

        $args = array_merge( $defaults, $args );

        // Determin image ID.
        if ( is_array( $image ) && isset( $image['ID'] ) ) {
            $id = $image['ID'];
        } else {
            $id = $image;
        }

        // Get full image source for checks.
        $full_src = wp_get_attachment_image_src( $id, 'full' );

        if ( ! $full_src ) {
            return;
        }

        $full_src = $full_src[0];

        // Get file extension.
        if ( function_exists( 'pathinfo' ) ) {
            $ext = strtolower( pathinfo( $full_src, PATHINFO_EXTENSION ) );
        } else {
            $last_dot_position = strrpos( $full_src, '.' );

            if ( $last_dot_position !== false ) {
                $ext = strtolower( substr( $full_src, $last_dot_position + 1 ) );
            } else {
                return;
            }
        }

        if ( $ext == 'svg' ) {
            $icon_path = get_attached_file( $id );
            $icon_file = file_get_contents( $icon_path );
            $position  = strpos( $icon_file, '<svg' );
            $icon      = substr( $icon_file, $position );
            $icon      = wp_kses( $icon, $allowed_svg_tags );
            $before    = '<figure class="pdg-svg">';
            $after     = '</figure>';

            echo $before . $icon . $after;
        } else {
            PDG::img( $image, $size, $args );
        }

    }

    public static function get_image_src( $image, $size = 'full' ) {

        // Determin image ID.
        if ( is_array( $image ) && isset( $image['ID'] ) ) {
            $id = $image['ID'];
        } else {
            $id = $image;
        }

        // Get full image source for checks.
        $full_src = wp_get_attachment_image_src( $id, 'full' );

        if ( ! $full_src ) {
            return;
        }

        $full_src = $full_src[0];

        // Get file extension.
        if ( function_exists( 'pathinfo' ) ) {
            $ext = strtolower( pathinfo( $full_src, PATHINFO_EXTENSION ) );
        } else {
            $last_dot_position = strrpos( $full_src, '.' );

            if ( $last_dot_position !== false ) {
                $ext = strtolower( substr( $full_src, $last_dot_position + 1 ) );
            } else {
                return;
            }
        }

        // Get image source.
        $wp_src = wp_get_attachment_image_src( $id, $size );
        $src    = $wp_src[0];
    
        return esc_url( $src );

    }

    public static function nav( $location, $class = '', $a_class = '', $depth = 0, $walker = '' ) {

        $items_wrap = '<ul>%3$s</ul>';
    
        if ( ! empty( $class ) ) {
            $items_wrap = '<ul class="' . $class . '">%3$s</ul>';
        }
    
        return wp_nav_menu(
            array(
                'theme_location'  => $location,
                'menu'            => '',
                'container'       => '',
                'container_class' => '',
                'container_id'    => '',
                'menu_class'      => 'menu',
                'menu_id'         => '',
                'echo'            => true,
                'fallback_cb'     => 'wp_page_menu',
                'before'          => '',
                'after'           => '',
                'link_before'     => '',
                'link_after'      => '',
                'items_wrap'      => $items_wrap,
                'depth'           => $depth,
                'walker'          => $walker,
                'add_a_class'     => $a_class
            )
        );
    }

    public static function languages( $style = 'default', $lang_display = 'code', $active_first = false ) {

        $hide = self::get_global_option( 'pdg_general_hide_langswitch' );

        if (
            ! function_exists( 'icl_get_languages' ) ||
            $hide == 'yes' ||
            ( $hide == 'logged' && ( ! is_user_logged_in() || ! current_user_can( 'administrator' ) ) )
        ) {
            return;
        }

        $mapping = apply_filters( 'pdg_language_mapping', [] );
    
        $args = [
            'current'      => '',
            'languages'    => icl_get_languages( 'skip_missing=0' ),
            'lang_display' => $lang_display,
            'mapping'      => $mapping
        ];
    
        // Get current language.
        foreach ( $args['languages'] as $lang ) {
            if ( $lang['active'] == 1 ) {
                $args['current'] = $lang;
            }
        }
    
        if ( is_array( $args['languages'] ) && count( $args['languages'] ) <= 1 ) {
            return;
        }
    
        if ( $style == 'default' ) {
            if ( $active_first ) {
                usort( $args['languages'], 'pdg_language_switcher_sort' );
            }
    
            get_template_part( 'partials/languages', null, $args );
        } elseif ( $style == 'dropdown' ) {
            get_template_part( 'partials/languages', 'dropdown', $args );
        }
    
    }

    public static function pager( $query = false, $arrows = [], $pages_on_each_side = 2, $show_dots = true, $show_inactive_arrows = false ) {

        $args = [
            'current_page' => 0,
            'total_pages'  => 0,
            'arrows'       => [],
            'pages'        => []
        ];



        // Get query.
        global $wp_query;

        if ( ! $query ) {
            $query = $wp_query;
        }

        // Get current page.
        global $paged;

        if ( empty( $paged ) ) {
            $paged = 1;
        }

        // Get total pages.
        $pages = $query->max_num_pages;

        if ( ! $pages ) {
            $pages = 1;
        }

        // Exit if page count is less then 2.
        if ( $pages <= 1 ) {
            return;
        }

        // Get start and end pages.
        $start_page = max( 1, $paged - $pages_on_each_side );
        $end_page   = min( $pages, $paged + $pages_on_each_side );

        // Adjust start page and end page if we need to show specified pages and we are near the start or end.
        while ( ( $end_page - $start_page + 1 ) < ( 2 * $pages_on_each_side + 1 ) && $start_page > 1 ) {
            $start_page--;
        }

        // Adjust end page if we need to show specified pages and we are near the end.
        while ( ( $end_page - $start_page + 1 ) < ( 2 * $pages_on_each_side + 1 ) && $end_page < $pages ) {
            $end_page++;
        }

        // Generate pager array.
        for ( $i = $start_page; $i <= $end_page; $i++ ) {
            $args['pages'][] = [
                'type'       => 'number',
                'value'      => $i,
                'url'        => get_pagenum_link( $i ),
                'is_current' => $i == $paged
            ];
        }

        // Add dots and the last page number if necessary.
        if ( $show_dots && $end_page < $pages && $paged < ( $pages - $pages_on_each_side ) ) {
            if ( $paged < ( $pages - $pages_on_each_side - 1 ) ) {
                $args['pages'][] = [
                    'type'       => 'dots',
                    'value'      => '...',
                    'is_current' => false
                ];
            }

            $args['pages'][] = [
                'type'       => 'number',
                'value'      => $pages,
                'url'        => get_pagenum_link( $pages ),
                'is_current' => $pages == $paged
            ];
        }



        // Set default arrows.
        $args['arrows'] = [
            'first' => [
                'value'     => '<span>&laquo;</span>',
                'url'       => get_pagenum_link( 1 ),
                'is_active' => $paged > 1
            ],
            'prev'  => [
                'value'     => '<span>&lsaquo;</span>',
                'url'       => get_pagenum_link( $paged - 1 ),
                'is_active' => $paged > 1
            ],
            'next'  => [
                'value'     => '<span>&rsaquo;</span>',
                'url'       => get_pagenum_link( $paged + 1 ),
                'is_active' => $paged < $pages
            ],
            'last'  => [
                'value'     => '<span>&raquo;</span>',
                'url'       => get_pagenum_link( $pages ),
                'is_active' => $paged < $pages
            ]
        ];

        if ( is_array( $arrows ) && ! empty( $arrows ) ) {
            if ( array_key_exists( 'first', $arrows ) ) {
                $args['arrows']['first']['value'] = $arrows['first'];
            } else {
                $args['arrows']['first'] = false;
            }

            if ( array_key_exists( 'prev', $arrows ) ) {
                $args['arrows']['prev']['value'] = $arrows['prev'];
            } else {
                $args['arrows']['prev'] = false;
            }

            if ( array_key_exists( 'next', $arrows ) ) {
                $args['arrows']['next']['value'] = $arrows['next'];
            } else {
                $args['arrows']['next'] = false;
            }

            if ( array_key_exists( 'last', $arrows ) ) {
                $args['arrows']['last']['value'] = $arrows['last'];
            } else {
                $args['arrows']['last'] = false;
            }
        }

        // Hide arrows if inactive is not shown.
        if ( ! $show_inactive_arrows ) {
            $args['arrows']['first'] = $paged > 1 ? $args['arrows']['first'] : false;
            $args['arrows']['prev']  = $paged > 1 ? $args['arrows']['prev'] : false;
            $args['arrows']['next']  = $paged < $pages ? $args['arrows']['next'] : false;
            $args['arrows']['last']  = $paged < $pages ? $args['arrows']['last'] : false;
        }



        // Render template.
        $args['current_page'] = $paged;
        $args['total_pages']  = $pages;

        get_template_part( 'partials/pager', null, $args );

    }

    public static function set_acf_language() {

        return acf_get_setting( 'default_language' );

    }

    public static function get_global_option( $name, $post_id = 'option' ) {

        add_filter( 'acf/settings/current_language', 'PDG::set_acf_language', 100 );
        $option = get_field( $name, $post_id );
        remove_filter( 'acf/settings/current_language', 'PDG::set_acf_language', 100 );
    
        return $option;

    }

    public static function get_current_url() {

        global $wp;
        
        if ( isset( $wp->request ) ) {
            return home_url( add_query_arg( [], $wp->request ) );
        }
        
        // Fallback with sanitization
        $protocol = is_ssl() ? 'https' : 'http';
        
        // Sanitize host
        $host = isset( $_SERVER['HTTP_HOST'] ) ? sanitize_text_field( wp_unslash( $_SERVER['HTTP_HOST'] ) ) : '';
        
        // Sanitize request URI
        $request_uri = isset( $_SERVER['REQUEST_URI'] ) ? esc_url_raw( wp_unslash( $_SERVER['REQUEST_URI'] ) ) : '';
        
        // Remove query string
        $parsed_url = parse_url( $request_uri );
        $path = isset( $parsed_url['path'] ) ? $parsed_url['path'] : '';
        
        return $protocol . '://' . $host . $path;
    
    }

    public static function get_phone( $value ) {

        if ( $value ) {
            $value = trim( $value );

            return [
                'label' => $value,
                'url'   => 'tel:' . str_replace( [' ', '(', ')'], '', $value )
            ];
        }

        return [];

    }

    public static function get_email( $value ) {

        if ( $value ) {
            $value = trim( $value );

            return [
                'label' => $value,
                'url'   => 'mailto:' . $value
            ];
        }

        return [];

    }

    public static function add_block( $name, $style = true, $script = false, $assets = false, $nestable = false, $mode = 'edit' ) {

        $slug = strtolower( str_replace( ' ', '-', $name ) );
        $slug = str_replace( ['(', ')'], '', $slug );
    
        $args = [
            'name'            => $slug,
            'title'           => $name,
            'icon'            => '<svg version="1.1" id="Layer_1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" x="0px" y="0px" viewBox="0 0 46.4 46.4" style="enable-background:new 0 0 46.4 46.4;" xml:space="preserve"><style type="text/css">.st0{opacity:0.2;fill:#252B31;enable-background:new    ;}.st1{fill:#252B31;}.st2{fill:#F6F6F6;}</style><circle class="st0" cx="23.2" cy="23.2" r="23.2"/><path class="st1" d="M37.7,12.6V9.9c0-1.5-1.2-2.8-2.8-2.8l0,0c-0.8,0-1.5,0.3-2,0.9c-2.9-1.9-6.3-2.8-9.7-2.8c-3.4,0-6.8,1-9.7,2.8c-0.5-0.6-1.3-0.9-2-0.9c-1.5,0-2.8,1.2-2.8,2.7l0,0v2.7c-2.2,3.1-3.5,6.8-3.5,10.6c0,9.9,8.1,18,18,18s18-8.1,18-18C41.2,19.4,39.9,15.7,37.7,12.6z M37.6,30.5c0,2.2-1.8,4.1-4.1,4.1l0,0h-7.4l2.2-5.5h2.1c0.8-0.2,1.5-0.9,1.7-1.7v-5.5c-0.2-1-1.1-1.7-2.2-1.8h-2.4c-1.2,0-2.2,1-2.2,2.3h-0.1v6.1h-4.3v-6.6c-0.2-1-1.1-1.7-2.2-1.8h-2.4c-1.2,0-2.2,1-2.2,2.3H14v4.5h0.1c0,1,0.7,2,1.7,2.2h1.9l2.1,5.5h-7.2c-2.2,0-4.1-1.8-4.1-4.1l0,0V14.8l5.5-3h18l5.4,3L37.6,30.5z"/><path class="st2" d="M14.2,11.8l-5.5,3v15.7c0,2.2,1.8,4.1,4.1,4.1l0,0H20l-2.1-5.5H16c-1-0.3-1.7-1.2-1.7-2.2h-0.1v-4.5h0.1c0-1.2,1-2.3,2.2-2.3h2.4c1,0,1.9,0.8,2.2,1.8v6.6h4.3v-6.1h0.1c0-1.2,1-2.3,2.2-2.3h2.4c1,0,1.9,0.8,2.2,1.8v5.5c-0.2,0.8-0.8,1.5-1.7,1.7h-2.1l-2.2,5.5h7.4c2.2,0,4.1-1.8,4.1-4.1l0,0V14.7l-5.4-3L14.2,11.8L14.2,11.8z"/></svg>',
            'mode'            => $mode,
            'render_template' => 'blocks/' . $slug . '/index.php',
            'category'        => 'custom',
            'keywords'        => [ str_replace( '-', ' ', $slug ) ],
            'supports'        => [
                'anchor' => true
            ]
        ];
    
        if ( ! is_admin() ) {
            if ( $assets && is_array( $assets ) ) {
                $args['enqueue_assets'] = function() use ( $assets, $style, $script, $slug ) {
                    foreach ( $assets as $asset ) {
                        if ( strpos( $asset, 'css:' ) === 0 ) {
                            wp_enqueue_style( str_replace( 'css:', '', $asset ) );
                        } elseif ( strpos( $asset, 'js:' ) === 0 ) {
                            wp_enqueue_script( str_replace( 'js:', '', $asset ) );
                        }
                    }

                    if ( $style ) {
                        $style_url = apply_filters( 'pdg_block_css_url', PDGC_URL . '/blocks/[block]/style.css' );
                        $style_url = str_replace( '[block]', $slug, $style_url );
            
                        wp_enqueue_style( "pdgc_{$slug}", $style_url, [], PDGC_VER );
                    }

                    if ( $script ) {
                        $script_url = apply_filters( 'pdg_block_js_url', PDGC_URL . '/blocks/[block]/script.js' );
                        $script_url = str_replace( '[block]', $slug, $script_url );
            
                        wp_enqueue_script( "pdgc_{$slug}", $script_url, ['jquery'], PDGC_VER, true );
                    }
                };
            } else {
                if ( $style ) {
                    $style_url = apply_filters( 'pdg_block_css_url', PDGC_URL . '/blocks/[block]/style.css' );
                    $style_url = str_replace( '[block]', $slug, $style_url );
        
                    $args['enqueue_style'] = $style_url;
                }
            
                if ( $script ) {
                    $script_url = apply_filters( 'pdg_block_js_url', PDGC_URL . '/blocks/[block]/script.js' );
                    $script_url = str_replace( '[block]', $slug, $script_url );
        
                    $args['enqueue_script'] = $script_url;
                }

                if ( $assets ) {
                    $args['enqueue_assets'] = $assets;
                }
            }
        }
    
        if ( $nestable === true ) {
            $args['mode']            = 'preview';
            $args['supports']['jsx'] = true;
        } elseif ( is_array( $nestable ) ) {
            $args['parent'] = $nestable;
        }
    
        acf_register_block_type( $args );

    }

    public static function is_odd( $number ) {

        return $number % 2 != 0;

    }

    public static function get_youtube_video_id( $url ) {

        parse_str( parse_url( $url, PHP_URL_QUERY ), $vars );

        if ( $vars && isset( $vars['v'] ) ) {
            return $vars['v'];
        } else {
            $arr = explode( '/', $url );
            return end( $arr );
        }

        return '';

    }

    public static function get_youtube_video_thumbnail( $id, $quality = 'max' ) {

        if ( strpos( $id, 'youtube' ) !== false || strpos( $id, 'youtu.be' ) !== false || strpos( $id, 'y2u.be' ) !== false ) {
            $id = self::get_youtube_video_id( $id );
        }

        if ( strpos( $id, 'http' ) !== false ) {
            return false;
        }

        $q = [
            'low'    => 'sddefault.jpg',
            'medium' => 'mqdefault.jpg',
            'high'   => 'hqdefault.jpg',
            'max'    => 'maxresdefault.jpg'
        ];

        return 'https://img.youtube.com/vi/' . $id . '/' . $q[$quality];

    }

    public static function icon( $name, $echo = true ) {

        $path = apply_filters( 'pdg_icon_path', 'svg' );

        if ( $echo ) {
            get_template_part( "{$path}/{$name}" );
        } else {
            ob_start();

            get_template_part( "{$path}/{$name}" );

            return ob_get_clean();
        }

    }

    public static function dump( $variable ) {

        echo '<pre>';
        var_dump( $variable );
        echo '</pre>';

    }

}