<?php if ( ! defined( 'ABSPATH' ) ) exit;

/**
 * Add JavaScript plugin files based on
 * options page  choices.
 * 
 * @since 3.0.0
 * @param string $field Option field name
 * @param string $ver   Plugin version
 * @param string $js    URL to plugins JavaScript file
 * @param string $css   URL to plugins CSS file
 * @param array  $args  Additional arguments
 */
function pdg_add_js_plugin( $field, $ver, $js = false, $css = false, $args = [] ) {

    $handle = 'pdg-' . str_replace( 'pdg_js_', '', $field );
    $value  = PDG::get_global_option( $field );

    // Prepare url for Google Maps.
    if ( $field == 'pdg_js_google_maps' ) {
        $api_key  = PDG::get_global_option( 'pdg_gm_key' );
        $callback = PDG::get_global_option( 'pdg_gm_callback' );

        if ( ! $api_key ) {
            return;
        }

        $js .= "?key={$api_key}";

        if ( $callback ) {
            $js .= "&callback={$callback}";
        }
    }

    $args = true;

    if ( $handle == 'pdg-google_maps' ) {
        $args = [
            'strategy' => 'async',
            'in_footer' => true
        ];
    }

    if ( $value && $value !== 'disable' ) {
        if ( $value == 'register' ) {
            if ( $js ) {
                wp_register_script( $handle, $js, ['jquery'], $ver, $args );

                if ( isset( $args['js'] ) && $args['js'] ) {
                    foreach ( $args['js'] as $i => $src ) {
                        wp_register_script( $handle . '-' . ( $i + 1 ), $src, ['jquery'], $ver, true );
                    }
                }
            }
    
            if ( $css ) {
                wp_register_style( $handle, $css, [], $ver );
            }
        } elseif ( $value == 'enqueue' ) {
            if ( $js ) {
                wp_enqueue_script( $handle, $js, ['jquery'], $ver, $args );

                if ( isset( $args['js'] ) && $args['js'] ) {
                    foreach ( $args['js'] as $i => $src ) {
                        wp_enqueue_script( $handle . '-' . ( $i + 1 ), $src, ['jquery'], $ver, true );
                    }
                }
            }
    
            if ( $css ) {
                wp_enqueue_style( $handle, $css, [], $ver );
            }
        }
    }

}

/**
 * Get script url by handle.
 * 
 * @since 3.0.0
 * @param string $handle Script handle
 * @return array/bool
 */
function pdg_get_script_src_by_handle( $handle ) {

    global $wp_scripts;

    if ( in_array( $handle, $wp_scripts->queue ) ) {
        $arr = array(
            'src'   => $wp_scripts->registered[$handle]->src,
            'ver'   => $wp_scripts->registered[$handle]->ver,
            'extra' => '',
            'deps'  => $wp_scripts->registered[$handle]->deps
        );

        if ( isset( $wp_scripts->registered[$handle]->extra['data'] ) ) {
            $arr['extra'] = $wp_scripts->registered[$handle]->extra['data'];
        }

        if ( isset( $wp_scripts->registered[$handle]->extra['before'] ) ) {
            if ( is_array( $wp_scripts->registered[$handle]->extra['before'] ) ) {
                foreach ( $wp_scripts->registered[$handle]->extra['before'] as $before ) {
                    $arr['extra'] .= $before;
                }
            } else {
                $arr['extra'] .= $wp_scripts->registered[$handle]->extra['before'];
            }
        }

        return $arr;
    } elseif ( isset( $wp_scripts->registered[$handle] ) ) {
        $arr = array(
            'src'   => $wp_scripts->registered[$handle]->src,
            'ver'   => $wp_scripts->registered[$handle]->ver,
            'extra' => '',
            'deps'  => $wp_scripts->registered[$handle]->deps
        );

        if ( isset( $wp_scripts->registered[$handle]->extra['data'] ) ) {
            $arr['extra'] = $wp_scripts->registered[$handle]->extra['data'];
        }

        if ( isset( $wp_scripts->registered[$handle]->extra['before'] ) ) {
            if ( is_array( $wp_scripts->registered[$handle]->extra['before'] ) ) {
                foreach ( $wp_scripts->registered[$handle]->extra['before'] as $before ) {
                    $arr['extra'] .= $before;
                }
            } else {
                $arr['extra'] .= $wp_scripts->registered[$handle]->extra['before'];
            }
        }

        if ( $arr['src'] ) {
            return $arr;
        }
    }

    return false;

}

/**
 * Get stylesheet url by handle.
 * 
 * @since 3.0.0
 * @param string $handle Script handle
 * @return array/bool
 */
function pdg_get_style_src_by_handle( $handle ) {

    global $wp_styles;

    if ( in_array( $handle, $wp_styles->queue ) ) {
        $arr = array(
            'src'  => $wp_styles->registered[$handle]->src,
            'ver'  => $wp_styles->registered[$handle]->ver,
            'deps' => $wp_styles->registered[$handle]->deps
        );

        return $arr;
    }

    return false;

}

/**
 * Add theme assets.
 */
add_action( 'wp_enqueue_scripts', function() {

    $assets = PDG_URL . '/assets';

    // Browser support.
    wp_enqueue_script( 'pdg-browser-support', "{$assets}/vendor/pandago/browser-support.js", [], false, true );

	wp_localize_script( 'pdg-browser-support', 'pdgBrowser', [
        'theme_url' => get_template_directory_uri(),
        'title'     => __( 'Lūdzu atjauniniet savu pārlūkprogrammu', 'pdg' ),
        'text1'     => __( 'Jūsu Tīmekļa pārlūks nespēj pilnvērtīgi attēlot šo lapu, jo šī lapa ir būvēta balstoties uz moderniem Tīmekļa standartiem. Lai apskatītu šo lapu, aicinām Jūs izvēlēties kādu no modernajiem Tīmekļa pārlūkiem, kas pieejami zemāk.', 'pdg' ),
        'text2'     => __( 'Bezmaksas pārlūkprogrammas  - visas pārlūkprogrammas nodrošina vienādas pamatfunkcijas un ir viegli lietojamas. Izvēlieties, kuru pārlūkprogrammu vēlaties lejupielādēt:', 'pdg' ),
        'close'     => __( 'Aizvērt', 'pdg' )
    ] );

    // Resets.
    wp_enqueue_style( 'pdg-reset', "{$assets}/css/main.css" );

    // JS plugins.
    pdg_add_js_plugin( 'pdg_js_aos', '2.0.0', PDG_URL . '/assets/vendor/aos/aos.js', PDG_URL . '/assets/vendor/aos/aos.css' );
    pdg_add_js_plugin( 'pdg_js_datepicker', '1.12.1', PDG_URL . '/assets/vendor/datepicker/jquery-ui.min.js', PDG_URL . '/assets/vendor/datepicker/jquery-ui.min.css' );
    pdg_add_js_plugin( 'pdg_js_fancybox', '3.5.7', PDG_URL . '/assets/vendor/fancybox/jquery.fancybox.min.js', PDG_URL . '/assets/vendor/fancybox/jquery.fancybox.min.css' );
    pdg_add_js_plugin( 'pdg_js_flexslider', '2.7.2', PDG_URL . '/assets/vendor/flexslider/jquery.flexslider-min.js', PDG_URL . '/assets/vendor/flexslider/flexslider.css' );
    pdg_add_js_plugin( 'pdg_js_google_maps', '3', 'https://maps.googleapis.com/maps/api/js' );
    pdg_add_js_plugin( 'pdg_js_hoverintent', '1.10.0', PDG_URL . '/assets/vendor/hoverintent/jquery.hoverIntent.min.js' );
    pdg_add_js_plugin( 'pdg_js_isonscreen', '', PDG_URL . '/assets/vendor/isonscreen/jquery.isonscreen.min.js' );
    pdg_add_js_plugin( 'pdg_js_leaflet', '1.9.4', PDG_URL . '/assets/vendor/leaflet/leaflet.js', PDG_URL . '/assets/vendor/leaflet/leaflet.css' );
    pdg_add_js_plugin( 'pdg_js_matchheight', '0.7.2', PDG_URL . '/assets/vendor/match-height/jquery.matchHeight-min.js' );
    pdg_add_js_plugin( 'pdg_js_nicescroll', '3.7.6', PDG_URL . '/assets/vendor/nicescroll/jquery.nicescroll.min.js' );
    pdg_add_js_plugin( 'pdg_js_slick', '1.8.1', PDG_URL . '/assets/vendor/slick/slick.min.js', PDG_URL . '/assets/vendor/slick/slick.css' );
    pdg_add_js_plugin( 'pdg_js_sumoselect', '3.0.2', PDG_URL . '/assets/vendor/sumoselect/jquery.sumoselect.min.js', PDG_URL . '/assets/vendor/sumoselect/sumoselect.min.css' );
    pdg_add_js_plugin( 'pdg_js_tooltipster', '4.2.8', PDG_URL . '/assets/vendor/tooltipster/tooltipster.bundle.min.js', PDG_URL . '/assets/vendor/tooltipster/tooltipster.bundle.min.css' );
    pdg_add_js_plugin( 'pdg_js_validation', '1.19.5', PDG_URL . '/assets/vendor/jquery-validation/jquery.validate.min.js', false, [
        'js' => [PDG_URL . '/assets/vendor/jquery-validation/additional-methods.min.js']
    ] );
    pdg_add_js_plugin( 'pdg_js_videojs', '8.9.0', PDG_URL . '/assets/vendor/videojs/video.min.js', PDG_URL . '/assets/vendor/videojs/video-js.min.css' );

}, 1 );

/**
 * Remove assets by handle.
 */
add_action( 'wp_enqueue_scripts', function() {

    // Remove JS.
    if ( $remove_js = PDG::get_global_option( 'pdg_optimisation_remove_js' ) ) {
        $remove_js = array_map( 'trim', explode( ',', $remove_js ) );

        if ( $remove_js ) {
            foreach ( $remove_js as $handle ) {
                wp_deregister_script( $handle );
                wp_dequeue_script( $handle );
            }
        }
    }

    // Remove CSS.
    if ( $remove_css = PDG::get_global_option( 'pdg_optimisation_remove_css' ) ) {
        $remove_css = array_map( 'trim', explode( ',', $remove_css ) );

        if ( $remove_css ) {
            foreach ( $remove_css as $handle ) {
                wp_deregister_style( $handle );
                wp_dequeue_style( $handle );
            }
        }
    }

}, 9999 );

/**
 * Late loaded assets.
 */
function pdg_get_late_scripts( $handles ) {

    $scripts = [];
    
    global $wp_scripts;
    
    $skip = ['jquery', 'wp-polyfill'];

    if ( $handles ) {
        foreach ( $handles as $handle ) {
            if ( in_array( $handle, $skip ) ) {
                continue;
            }
    
            if ( isset( $wp_scripts->registered[$handle] ) ) {
                $script = $wp_scripts->registered[$handle];
    
                $script_item = [
                    'handle' => $handle,
                    'url'    => $script->src,
                    'ver'    => $script->ver,
                    'deps'   => $script->deps,
                    'extra'  => isset( $script->extra['data'] ) ? $script->extra['data'] : ''
                ];

                if ( isset( $script->extra['before'] ) ) {
                    if ( is_array( $script->extra['before'] ) ) {
                        foreach ( $script->extra['before'] as $before ) {
                            if ( ! $before ) continue;

                            $script_item['extra'] .= $before;
                        }
                    } else {
                        $script_item['extra'] .= $script->extra['before'];
                    }
                }

                $scripts[] = $script_item;

                wp_deregister_script( $handle );
                wp_dequeue_script( $handle );
                
                if ( ! empty( $script->deps ) ) {
                    $dependency_scripts = pdg_get_late_scripts( $script->deps );
                    
                    $scripts = array_merge( $dependency_scripts, $scripts );
                }
            }
        }
    }
    
    return $scripts;

}

add_action( 'wp_enqueue_scripts', function() {

    if ( isset( $_GET['pdg-no-lazy'] ) ) {
        return;
    }

    $late_load = [
        'js'  => [],
        'css' => []
    ];

    if ( $late_js = PDG::get_global_option( 'pdg_optimisation_late_js' ) ) {
        $late_load['js'] = array_map( 'trim', explode( ',', $late_js ) );
    }

    if ( $late_css = PDG::get_global_option( 'pdg_optimisation_late_css' ) ) {
        $late_load['css'] = array_map( 'trim', explode( ',', $late_css ) );
    }



    $late = [
        'js'    => [],
        'css'   => [],
        'extra' => [],
        'should_emit_content_loaded_event' => false
    ];

    // Form validation late loaded assets.
    $form = get_field( 'pdg_general_main_cf', 'option' );

    if ( class_exists( 'WPCF7' ) && $form ) {
        wp_enqueue_style( 'pdg-form-validation', PDG_URL . '/assets/vendor/pandago/form-validation.css', ['domelement:.js-pdg-form'] );
        wp_enqueue_script( 'pdg-form-validation', PDG_URL . '/assets/vendor/pandago/form-validation.js', ['jquery', 'domelement:.js-pdg-form'], false, true );

        wp_localize_script( 'pdg-form-validation', 'pdgFormValidation', WPCF7_ContactForm::get_instance( $form->ID )->get_properties()['messages'] );

        $late_load['js'][]  = 'pdg-form-validation';
        $late_load['css'][] = 'pdg-form-validation';
    }

    // Let developer load script immediately if necessary on some pages.
    $late_load['js'] = apply_filters( 'pdg_late_load_js', $late_load['js'] );

    // Push all scripts & styles.
    $late['js'] = pdg_get_late_scripts( $late_load['js'] );

    // Get extras.
    if ( $late['js'] ) {
        foreach ( $late['js'] as $script ) {
            if ( $script['extra'] ) {
                $late['extra'][] = [
                    'handle' => $script['handle'],
                    'data'   => $script['extra']
                ];
            }
        }
    }

    if ( isset( $late_load['css'] ) && ! empty( $late_load['css'] ) ) {
        // Reverse array order so that style order is preserved.
        $late_load['css'] = array_reverse( $late_load['css'] );

        // Let developer load script immediately if necessary on some pages.
        $late_load['css'] = apply_filters( 'pdg_late_load_css', $late_load['css'] );

        foreach ( $late_load['css'] as $handle ) {
            $style = pdg_get_style_src_by_handle( $handle );

            if ( ! $style ) {
                continue;
            }

            $late['css'][] = [
                'handle' => $handle,
                'url'    => $style['src'],
                'ver'    => $style['ver'],
                'deps'   => $style['deps']
            ];

            wp_deregister_style( $handle );
            wp_dequeue_style( $handle );
        }
    }

    if ( $late['js'] || $late['css'] ) {
        wp_enqueue_script( 'pdg-late-loader', PDG_URL . '/assets/vendor/pandago/late-loader.js', ['jquery'], false, true );

        wp_add_inline_script( 'pdg-late-loader', 'var pdgLateLoader = ' . json_encode( $late ), 'before' );
    }

}, 90 );

/**
 * Add admin assets.
 */
add_action( 'admin_enqueue_scripts', function() {

    wp_enqueue_style( 'pdg-admin-main', PDG_URL . '/assets/vendor/pandago/admin.css' );

} );

/**
 * Helper function for pdg_dump_assets which retrieves
 * all registered assets and returns them.
 * 
 * @since 3.0.0
 * @return array
 */
function pdg_get_dump_assets() {

    global $wp_scripts, $wp_styles;

    $result = [
        'scripts' => [],
        'styles'  => []
    ];
    
    foreach( $wp_scripts->queue as $handle ) {
        $result['scripts'][$handle] = $wp_scripts->registered[$handle]->src . ';';
    }

    foreach( $wp_styles->queue as $handle ) {
        $result['styles'][$handle] = $wp_styles->registered[$handle]->src . ';';
    }

    return $result;

}

/**
 * Checks if user is logged in and 'dump-assets'
 * parameter is present.
 * 
 * @since 3.0.0
 */
function pdg_dump_assets() {

    if ( isset( $_GET['dump-assets'] ) && is_user_logged_in() ) {
        echo '<pre>';
        var_dump( pdg_get_dump_assets() );
        echo '</pre>';

        die();
    }

}
add_action( 'wp_head', 'pdg_dump_assets');